<?php
// File: keybin_webhook.php

require_once('config.php'); // Ensure this path is correct for your OpenCart installation

// Use OpenCart DB constants from config.php
$db_host = DB_HOSTNAME;
$db_name = DB_DATABASE;
$db_user = DB_USERNAME;
$db_pass = DB_PASSWORD;
$db_prefix = DB_PREFIX;

// Set up error logging to a file in the current directory
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/keybin_create_listing_webhook_log.txt');

// Log flags
$loggingEnabled = true;

// Custom logging function
function custom_log($message) {
    global $loggingEnabled;
    if ($loggingEnabled) {
        error_log($message); // Call the built-in error_log function
    }
}


// Margins

$marginsFilePath = __DIR__ . '/keybin-webhook-margins.txt';
$globalMargin = 0;
$categoryMargins = [];

// Function to load margins from the text file
function loadMargins($filePath) {
    $globalMargin = 0;
    $categoryMargins = [];

    if (file_exists($filePath)) {
        $lines = file($filePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        
        foreach ($lines as $line) {
             // Trim line and skip if it's a comment
             $line = trim($line);
             if (empty($line) || strpos($line, '#') === 0 || strpos($line, '//') === 0) {
                 continue; // Skip comments and empty lines
             }
            $parts = explode(';', trim($line));
            if (count($parts) == 3) {
                // Determine if it's global or category margin
                if (strtolower($parts[0]) === "global") {
                    $globalMargin = floatval($parts[2]);
                } else {
                    // Add category margin to the array using CategoryID as key
                    $categoryMargins[$parts[0]] = floatval($parts[2]); // Use CategoryID as key
                }
            }
        }
    } else {
        custom_log("Margins file not found: " . $filePath);
    }

    return [$globalMargin, $categoryMargins];
}

list($globalMargin, $categoryMargins) = loadMargins($marginsFilePath);

// End margins

// Start time for performance monitoring
$startTime = microtime(true);

// Establish database connection
$mysqli = new mysqli($db_host, $db_user, $db_pass, $db_name);
if ($mysqli->connect_error) {
    custom_log("Database connection failed: " . $mysqli->connect_error);
    die("Database connection failed: " . $mysqli->connect_error);
}

// Capture webhook headers
$headers = getallheaders();
custom_log("Received Headers: " . print_r($headers, true));

// Access the event header correctly (case-sensitive)
$event = '';
foreach ($headers as $key => $value) {
    if (strtolower($key) === 'x-keybin-event') {
        $event = trim($value);
        break;
    }
}
custom_log("All headers received:\n" . print_r($headers, true));


if (empty($event)) {
    custom_log("X-Keybin-Event header missing. Headers received:\n" . print_r($headers, true));
    http_response_code(400); // Bad request
    echo json_encode(['status' => 'error', 'message' => 'Missing required X-Keybin-Event header']);
    exit;
}

// Log the raw request body
$rawBody = file_get_contents('php://input');
custom_log("Raw Request Body: " . $rawBody);





// Check if the request body is empty and handle it gracefully
if (empty($rawBody) || $rawBody === "[]") {
    custom_log("Empty or invalid JSON received for event: " . $event);
    http_response_code(200);
    echo json_encode(['status' => 'success', 'message' => 'Empty body received, no action taken']);
    exit;
}





// Get the raw POST data
$rawRequestBody = file_get_contents('php://input');

// Log the raw request body for debugging
custom_log("Raw Request Body: $rawRequestBody");

// Decode the JSON safely
$data = json_decode($rawRequestBody, true);

// Check for JSON decoding errors
if (json_last_error() !== JSON_ERROR_NONE) {
    custom_log("JSON decode error: " . json_last_error_msg());
    // Optionally return a 400 Bad Request response
    http_response_code(400);
    echo json_encode(["error" => "Invalid JSON"]);
    exit; // Stop execution
}

// Accessing fields with default values if missing
$model = $data['product']['id'] ?? null; // Defaults to null if not set
$activeStock = $data['activeStock'] ?? 0; // Defaults to 0 if not set
$price = $data['price']['price'] ?? null; // Defaults to null if not set
$listingId = $data['id'] ?? null; // Defaults to null if not set
$isActive = $data['isActive'] ?? false; // Defaults to false if not set

// Log extracted values for debugging
custom_log("Extracted values | MODEL: $model | Active Stock: $activeStock | Price: $price | Listing ID: $listingId | Is Active: $isActive");

// Further processing logic...

// Further processing...






  // Prepare SQL to check if the product exists in the OpenCart database by model
    $stmt = $mysqli->prepare("
    SELECT p.product_id, p.keybin_listing, p.quantity, p.price, p.keybin_disable_price, p.keybin_disable_stock, 
    pc.category_id 
    FROM " . $db_prefix . "product p
    LEFT JOIN " . $db_prefix . "product_to_category pc ON p.product_id = pc.product_id
    WHERE p.model = ?
    LIMIT 1
    ");
    $stmt->bind_param("s", $model);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 0) {
    custom_log("Product MODEL not found for event: " . $event . " | Model: " . $model);
    return;
    }

    // Fetch the product details
    $product = $result->fetch_assoc();
    $productId = $product['product_id'];
    $currentStock = $product['quantity'];
    $currentPrice = $product['price'];
    $currentListingId = $product['keybin_listing'];
    $disablePrice = $product['keybin_disable_price'];
    $disableStock = $product['keybin_disable_stock'];

    // Get category ID (ensure this is fetched from the product correctly)
    $categoryId = $product['category_id']; // This is the ID you want for margin calculations
    if (empty($categoryId)) {
    custom_log("No category ID found for MODEL: " . $model);
    return; // Skip processing if no category ID
    }


    // Log product ID and flags
    custom_log("Found Product ID: " . $productId . " for MODEL: " . $model . " | Disable Price: $disablePrice | Disable Stock: $disableStock Category ID: " . $categoryId);

    // Determine if updates should occur
    $updateStock = ($disableStock == 0);
    $updatePrice = ($disablePrice == 0);
    
    if ($listingId === $currentListingId) {
        // Matching Listing ID logic
        if (!$isActive) {
            // If isActive is false, set stock to 0
            $activeStock = 0;
            custom_log("Setting stock to 0 for MODEL: $model because isActive is false.");
        }

        // Check for current price being 0
        if ($currentPrice == 0) {
            custom_log("Current price for MODEL: $model is 0, updating stock, price, and keybin_listing.");

              // Calculate price with margins
    $priceWithMargin = $price * (1 + ($globalMargin / 100));
    if (isset($categoryMargins[$categoryId])) {
        $priceWithMargin *= (1 + ($categoryMargins[$categoryId] / 100));
    }
            // Update stock
            if ($updateStock) {
                $stmtUpdateStock = $mysqli->prepare("UPDATE " . $db_prefix . "product SET quantity = ? WHERE product_id = ?");
                $stmtUpdateStock->bind_param("ii", $activeStock, $productId);
                $stmtUpdateStock->execute();
                custom_log("Updated stock for MODEL: " . $model . " to " . $activeStock);
                $stmtUpdateStock->close();
            } else {
                custom_log("Stock update skipped for MODEL: " . $model . " because keybin_disable_stock is set.");
            }

            // Update price
            if ($updatePrice) {
                $stmtUpdatePrice = $mysqli->prepare("UPDATE " . $db_prefix . "product SET price = ?, keybin_listing = ? WHERE product_id = ?");
                $stmtUpdatePrice->bind_param("dsi", $price, $listingId, $productId);
                $stmtUpdatePrice->execute();
                custom_log("Updated price for MODEL: " . $model . " to " . $price);
                $stmtUpdatePrice->close();
            } else {
                custom_log("Price update skipped for MODEL: " . $model . " because keybin_disable_price is set.");
            }

            // Update keybin_listing
            $stmtUpdateListing = $mysqli->prepare("UPDATE " . $db_prefix . "product SET keybin_listing = ? WHERE product_id = ?");
            $stmtUpdateListing->bind_param("si", $listingId, $productId);
            $stmtUpdateListing->execute();
            custom_log("Updated keybin_listing for MODEL: " . $model . " to " . $listingId);
            $stmtUpdateListing->close();
        } else {
            // If the price is not 0 and isActive is true
            if ($isActive) {
                // Handle updates based on active state
                if ($activeStock < 0) {
                    custom_log("Invalid active stock for MODEL: $model. Skipping.");
                    return;
                }

                // Update stock if allowed
                if ($updateStock) {
                    $stmtUpdateStock = $mysqli->prepare("UPDATE " . $db_prefix . "product SET quantity = ? WHERE product_id = ?");
                    $stmtUpdateStock->bind_param("ii", $activeStock, $productId);
                    $stmtUpdateStock->execute();
                    custom_log("Updated stock for MODEL: " . $model . " to " . $activeStock);
                    $stmtUpdateStock->close();
                } else {
                    custom_log("Stock update skipped for MODEL: " . $model . " because keybin_disable_stock is set.");
                }

                // Update price if allowed
                if ($updatePrice) {

                // PRICES MARGIN FROM FILE | Check for global margin and category margin | EDIT HERE TO CRETE LOGIC FOR PRICES
                $priceWithMargin = $price * (1 + ($globalMargin / 100)); // Apply global margin

                // Get category ID (ensure this is fetched from the product correctly)
                $categoryId = $product['category_id']; // Make sure 'category_id' exists in your product fetch

                 // Apply category-specific margin if it exists
              if (isset($categoryMargins[$categoryId])) {
                $priceWithMargin *= (1 + ($categoryMargins[$categoryId] / 100)); // Apply category margin
                  } else {
                    custom_log("No category margin found for Category ID: " . $categoryId);
                }

                    // END PRICES MARGIN FROM FILE
                    /* USED IF WE ADD FIXED MARGIN (change also above)
                    $priceWithMargin = $price + $globalMargin;
                    if (isset($categoryMargins[$product['category_id']])) {
                        $priceWithMargin += $categoryMargins[$product['category_id']];
                    }
                    */
                    $stmtUpdatePrice = $mysqli->prepare("UPDATE " . $db_prefix . "product SET price = ?, keybin_listing = ? WHERE product_id = ?");
                    $stmtUpdatePrice->bind_param("dsi", $priceWithMargin, $listingId, $productId);
                    $stmtUpdatePrice->execute();
                    custom_log("Updated price for MODEL: " . $model . " to " . $priceWithMargin);
                    $stmtUpdatePrice->close();
                } else {
                    custom_log("Price update skipped for MODEL: " . $model . " because keybin_disable_price is set.");
                }
            }
        }
    } else {
        // Non-matching Listing ID logic
        if ($isActive) {
            if ($currentPrice == 0) {
                // If current price is 0, update price, stock, and listing
                custom_log("Current price is 0 for MODEL: " . $model . ", updating stock, price, and keybin_listing.");
                  // Calculate price with margins
                  $priceWithMargin = $price * (1 + ($globalMargin / 100));
                  if (isset($categoryMargins[$categoryId])) {
                      $priceWithMargin *= (1 + ($categoryMargins[$categoryId] / 100));
                  }
                // Update stock
                if ($updateStock) {
                    $stmtUpdateStock = $mysqli->prepare("UPDATE " . $db_prefix . "product SET quantity = ? WHERE product_id = ?");
                    $stmtUpdateStock->bind_param("ii", $activeStock, $productId);
                    $stmtUpdateStock->execute();
                    custom_log("Updated stock for MODEL: " . $model . " to " . $activeStock);
                    $stmtUpdateStock->close();
                } else {
                    custom_log("Stock update skipped for MODEL: " . $model . " because keybin_disable_stock is set.");
                }

                // Update price
                if ($updatePrice) {
                    $stmtUpdatePrice = $mysqli->prepare("UPDATE " . $db_prefix . "product SET price = ?, keybin_listing = ? WHERE product_id = ?");
                    $stmtUpdatePrice->bind_param("dsi", $price, $listingId, $productId);
                    $stmtUpdatePrice->execute();
                    custom_log("Updated price for MODEL: " . $model . " to " . $price);
                    $stmtUpdatePrice->close();
                } else {
                    custom_log("Price update skipped for MODEL: " . $model . " because keybin_disable_price is set.");
                }

                // Update keybin_listing
                $stmtUpdateListing = $mysqli->prepare("UPDATE " . $db_prefix . "product SET keybin_listing = ? WHERE product_id = ?");
                $stmtUpdateListing->bind_param("si", $listingId, $productId);
                $stmtUpdateListing->execute();
                custom_log("Updated keybin_listing for MODEL: " . $model . " to " . $listingId);
                $stmtUpdateListing->close();
            } else {
                // Calculate price with margins
                $priceWithMargin = $price * (1 + ($globalMargin / 100));
                if (isset($categoryMargins[$categoryId])) {
                    $priceWithMargin *= (1 + ($categoryMargins[$categoryId] / 100));
                }

                // Change this condition to update if price with margin is the same or lower
                if ($activeStock > 0 && $priceWithMargin <= $currentPrice) {
                    // Update stock, price, and listing ID
                    custom_log("Active stock is greater than 0 and new price with margin is same or lower than current price for MODEL: " . $model . ", updating stock, price, and keybin_listing.");

                    // Update stock
                    if ($updateStock) {
                        $stmtUpdateStock = $mysqli->prepare("UPDATE " . $db_prefix . "product SET quantity = ? WHERE product_id = ?");
                        $stmtUpdateStock->bind_param("ii", $activeStock, $productId);
                        $stmtUpdateStock->execute();
                        custom_log("Updated stock for MODEL: " . $model . " to " . $activeStock);
                        $stmtUpdateStock->close();
                    } else {
                        custom_log("Stock update skipped for MODEL: " . $model . " because keybin_disable_stock is set.");
                    }

                    // Update price
                    if ($updatePrice) {
                        $stmtUpdatePrice = $mysqli->prepare("UPDATE " . $db_prefix . "product SET price = ?, keybin_listing = ? WHERE product_id = ?");
                        $stmtUpdatePrice->bind_param("dsi", $priceWithMargin, $listingId, $productId);
                        $stmtUpdatePrice->execute();
                        custom_log("Updated price for MODEL: " . $model . " to " . $priceWithMargin);
                        $stmtUpdatePrice->close();
                    } else {
                        custom_log("Price update skipped for MODEL: " . $model . " because keybin_disable_price is set.");
                    }

                    // Update keybin_listing
                    $stmtUpdateListing = $mysqli->prepare("UPDATE " . $db_prefix . "product SET keybin_listing = ? WHERE product_id = ?");
                    $stmtUpdateListing->bind_param("si", $listingId, $productId);
                    $stmtUpdateListing->execute();
                    custom_log("Updated keybin_listing for MODEL: " . $model . " to " . $listingId);
                    $stmtUpdateListing->close();
                } else {
                    custom_log("Conditions for updating stock and price not met for MODEL: " . $model);
                }
            }
        } else {
            custom_log("Listing ID mismatch for MODEL: $model | Current Listing ID: $currentListingId | Incoming Listing ID: $listingId");
        }
    }


// Close the database connection
$mysqli->close();

// Calculate total execution time
$endTime = microtime(true);
$executionTime = $endTime - $startTime;
custom_log("Script execution time: " . $executionTime . " seconds");

http_response_code(200);
echo json_encode(['status' => 'success']);
?>