<?php
class ModelToolKeybin extends Model {
  
	public function checkCartPrices() {
    // Check price and stock only if parameter is enabled
    /*
    if (!$this->config->get('keybin_checkprice')) {
      return false;
    }
    */
    
    if (!isset($this->session->data['order_id'])) {
      return false;
    }
    
    if ($this->config->get('keybin_logs')) {
      file_put_contents(DIR_LOGS.'keybin.log', PHP_EOL . '>> ' . date('d/m/Y H:i:s'). ' - checkCartPrices' . PHP_EOL, FILE_APPEND | LOCK_EX);
    }
    
    $error = false;
    $updatePriceIfLower = false;
    
    // Get current product data
    $this->load->model('checkout/order');
    $order_products = $this->model_checkout_order->getOrderProducts($this->session->data['order_id']);
    
    if ($this->config->get('keybin_margin')) {
      $margin = ($this->config->get('keybin_margin') / 100) + 1;
    } else {
      $margin = 1;
    }
    
    foreach ($order_products as $product) {
      // check if current product is part of keybin catalog
      if (!$this->isKeybinProduct($product['product_id'])) {
        if ($this->config->get('keybin_logs')) {
          file_put_contents(DIR_LOGS.'keybin.log', 'Product ID: '.$product['product_id'].' is not detected as keybin product' . PHP_EOL, FILE_APPEND | LOCK_EX);
        }
        
        continue;
      }
      
      $product['price'];
      $product['quantity'];
      
      // Check cache for product listings (2 minutes cache)
      $cacheFile = DIR_CACHE . 'keybin_product_' . $product['model'] . '.cache';
      $cacheTimeout = 120; // 2 minutes for product listings
      $response = null;
      $usedCache = false;
      
      if (file_exists($cacheFile)) {
        $cacheData = json_decode(file_get_contents($cacheFile), true);
        
        if ($cacheData && isset($cacheData['timestamp']) && isset($cacheData['data'])) {
          $cacheAge = time() - $cacheData['timestamp'];
          
          if ($cacheAge < $cacheTimeout) {
            $response = $cacheData['data'];
            $usedCache = true;
            
            if ($this->config->get('keybin_logs')) {
              $timeLeft = $cacheTimeout - $cacheAge;
              file_put_contents(DIR_LOGS.'keybin.log', 'Using cached listings for product '.$product['model'].' (expires in '.$timeLeft.' seconds)' . PHP_EOL, FILE_APPEND | LOCK_EX);
            }
          }
        }
      }
      
      // If no valid cache, connect to API to get corresponding data in real-time
      if (!$response) {
        $response = $this->api('products/'.$product['model'].'/listings');
        
        // Log the API response for debugging
        if ($this->config->get('keybin_logs')) {
          file_put_contents(DIR_LOGS.'keybin.log', 'API Response for product '.$product['model'].': ' . print_r($response, true) . PHP_EOL, FILE_APPEND | LOCK_EX);
        }
        
        // Handle API throttle error (429)
        if (isset($response['statusCode']) && $response['statusCode'] == 429) {
          if ($this->config->get('keybin_logs')) {
            file_put_contents(DIR_LOGS.'keybin.log', 'API throttled (429) for product '.$product['model'].' - attempting to use stale cache' . PHP_EOL, FILE_APPEND | LOCK_EX);
          }
          
          // Try to use stale cache if available
          if (file_exists($cacheFile)) {
            $cacheData = json_decode(file_get_contents($cacheFile), true);
            if ($cacheData && isset($cacheData['data'])) {
              $response = $cacheData['data'];
              $usedCache = true;
              
              if ($this->config->get('keybin_logs')) {
                file_put_contents(DIR_LOGS.'keybin.log', 'Using stale cache for product '.$product['model'].' - Age: ' . (time() - $cacheData['timestamp']) . 's' . PHP_EOL, FILE_APPEND | LOCK_EX);
              }
            } else {
              continue; // Skip this product if no cache available
            }
          } else {
            continue; // Skip this product if no cache available
          }
        } else if (!empty($response) && isset($response['data'])) {
          // Cache the successful response
          $cacheData = array(
            'timestamp' => time(),
            'data' => $response
          );
          file_put_contents($cacheFile, json_encode($cacheData));
          
          if ($this->config->get('keybin_logs')) {
            file_put_contents(DIR_LOGS.'keybin.log', 'Cached listings for product '.$product['model'] . PHP_EOL, FILE_APPEND | LOCK_EX);
          }
        }
      }
      
      if (!empty($response) && isset($response['data']) && is_array($response['data'])) {
        $currentValues = array(
          'id' => '',
          'price' => '',
          'tierPrice' => '',
          'tierPrice2' => '',
          'minOrderQty' => '',
          'currency' => '',
          'activeStock' => 0,
        );
        
        // Check if all is ok
        foreach ($response['data'] as $r) {
          // skip if not active or no stock
          //if (empty($r['isActive']) || empty($r['activeStock'])) continue;
          if (empty($r['isActive']) || empty($r['activeStock']) || $r['activeStock'] <= 1) continue;

          $priceWithMargin = $r['price']['price'] * $margin;
          
          // Set lowest price
          if (!$currentValues['price']) {
            $currentValues['id'] = $r['id'];
            $currentValues['price'] = $priceWithMargin;
            $currentValues['activeStock'] = $r['activeStock'];
          } else if ($priceWithMargin < $currentValues['price']) {
            $currentValues['id'] = $r['id'];
            $currentValues['price'] = $priceWithMargin;
            $currentValues['activeStock'] = $r['activeStock'];
          }
        }
        
        // Price have changed?
        if ($this->config->get('keybin_logs')) {
          file_put_contents(DIR_LOGS.'keybin.log', 'Check price - Product ID: '.$product['product_id'].' - Listing: '.$currentValues['id'].' - Old: '.(float)$product['price'].' - New: '.(float)$currentValues['price'] . PHP_EOL, FILE_APPEND | LOCK_EX);
        }
        
        if ($currentValues['price'] > 0) {
          if (round($currentValues['price'], 2) !== round($product['price'], 2)) {
            $newPriceIsGreater = round($currentValues['price'], 2) > round($product['price'], 2);
            
            // always update if price is greater, and update if lower only if $updatePriceIfLower is enabled
            if ($updatePriceIfLower || $newPriceIsGreater) {
              // price have changed, update product with new price
              $this->db->query("UPDATE " . DB_PREFIX . "product SET price = '" . (float)$currentValues['price'] . "', date_modified = NOW() WHERE product_id = '" . (int)$product['product_id'] . "'");
              
              if ($this->config->get('keybin_logs')) {
                file_put_contents(DIR_LOGS.'keybin.log', 'Price changed, update to new price' . PHP_EOL, FILE_APPEND | LOCK_EX);
              }
            }
            
            // return error only if new price is greater
            if ($newPriceIsGreater) {
              $this->load->language('tool/keybin');
              $error = sprintf(
                  $this->language->get('text_keybin_price_changed'), 
                  $product['name'], 
                  number_format($product['price'], 1), // Format to 1 decimal place
                  number_format($currentValues['price'], 1) // Format to 1 decimal place
              );
          }
          }
        }
        
        // save current listing ID to be able to use it when validating the order
        $this->db->query("UPDATE " . DB_PREFIX . "product SET `keybin_listing` = '" . $this->db->escape($currentValues['id']) . "' WHERE product_id = '" . (int)$product['product_id'] . "'");
        
        if ($this->config->get('keybin_logs')) {
          file_put_contents(DIR_LOGS.'keybin.log', 'Set listing - Product ID: '.$product['product_id'].' - Listing: '.$currentValues['id'] . ' - Stock: '.(int)$currentValues['activeStock'] . PHP_EOL, FILE_APPEND | LOCK_EX);
        }
        
        // Check if there's enough stock for the current cart quantity
        if ($currentValues['activeStock'] < $product['quantity']) {
          // Stock is insufficient, update product with new stock
          $this->db->query("UPDATE " . DB_PREFIX . "product SET quantity = '" . (int)$currentValues['activeStock'] . "', date_modified = NOW() WHERE product_id = '" . (int)$product['product_id'] . "'");
          
          if ($this->config->get('keybin_logs')) {
            file_put_contents(DIR_LOGS.'keybin.log', 'Insufficient stock - Product ID: '.$product['product_id'].' - Required: '.(int)$product['quantity'].' - Available: '.(int)$currentValues['activeStock'] . ' - Stock updated in DB' . PHP_EOL, FILE_APPEND | LOCK_EX);
          }
          
          // return error
          $error = 'The product <b>'.$product['name'].'</b> is out of stock';
        }
      } else {
        // API didn't return valid data
        if ($this->config->get('keybin_logs')) {
          file_put_contents(DIR_LOGS.'keybin.log', 'API did not return valid data for product '.$product['model'].' (Product ID: '.$product['product_id'].')' . PHP_EOL, FILE_APPEND | LOCK_EX);
        }
      }
    }
    
    return $error;
	}
  
  public function processOrder($order_id, $order_products) {
    if ($this->config->get('keybin_logs')) {
      file_put_contents(DIR_LOGS.'keybin.log', PHP_EOL . '>> ' . date('d/m/Y H:i:s'). ' - processOrder' . PHP_EOL, FILE_APPEND | LOCK_EX);
    }
    
    // check cart price to make sure items have stock and are active
    $cartError = $this->checkCartPrices();
    
    if (!empty($cartError)) {
      error_log($cartError);
      
      return [
        'status' => 'error',
        'text' => $cartError,
      ];
    }
    
    $params = array();
    
    // prepare data for submiting orders
    foreach ($order_products as $i => $product) {
      // check if product already have a key
      if (!empty($product['keybin_key'])) {
        continue;
      }
      
      // check if current product is part of keybin catalog
      if (!$listingId = $this->isKeybinProduct($product['product_id'], true)) {
        if ($this->config->get('keybin_logs')) {
          file_put_contents(DIR_LOGS.'keybin.log', 'This product is not detected as keybin one - Product ID: '.$product['product_id'] . PHP_EOL, FILE_APPEND | LOCK_EX);
        }
        
        continue;
      }
      
      if ($this->config->get('keybin_margin')) {
        $margin = ($this->config->get('keybin_margin') / 100) + 1;
      } else {
        $margin = 1;
      }
      
      $params['items'][] = array(
        'listingId' => $listingId,
        'qty' => (int) $product['quantity'],
        'price' => (float) $product['price'] / $margin,
        'maxPrice' => (float) $product['price'],
      );
    }
    
    // if no keybin products to handle, cancel processing
    if (empty($params)) {
      if ($this->config->get('keybin_logs')) {
        file_put_contents(DIR_LOGS.'keybin.log', 'No keybin products to handle, cancel processing' . PHP_EOL, FILE_APPEND | LOCK_EX);
      }
      
      return [
        'status' => 'no_keybin',
        'text' => '',
      ];
    }
    
    // set order id as customReference
    $params['customReference'] = $product['order_id'];
    
    
    // submit orders to api
    $response = $this->api('orders', $params);
    
    // action on error
    if (!empty($response['status']) && ($response['status'] != 'complete' && $response['status'] != 'pending' && $response['status'] != 'test')) {
      error_log($response['message']);
      
      return [
        'status' => 'error',
        'text' => $response['message'],
      ];
    }
    
    // save keybin order id
    
    if (!empty($response['id'])) {
      $this->db->query("UPDATE `" . DB_PREFIX . "order` SET `keybin_id` = '".$this->db->escape($response['id'])."' WHERE `order_id` = '".(int) $order_id."'");
      
      if ($this->config->get('keybin_logs')) {
        file_put_contents(DIR_LOGS.'keybin.log', 'Order saved successfully - Keybin order ID: '.$response['id'] . PHP_EOL, FILE_APPEND | LOCK_EX);
      }
      
      // check if keys are ready
      $allKeysReady = true;
      
      $keysQuery = $this->api('orders/'.$response['id'].'/keys');
      
      $keysString = '';
      
      if (!empty($keysQuery) && empty($keysQuery['message'])) {
        foreach ($keysQuery as $item) {
          if (!empty($item['key'])) {
            if (substr($item['key'], 0, 5) == 'data:') {
              $keysString .= ($keysString ? "<br><br>" : '') . $item['product']['name'] . '<br><img src="'.$item['key'].'" style="max-width:100%"/>';
              // $keysString .= ($keysString ? "\n" : '') . $item['product']['name'] . ' - Image key, please check it in your account';
            } else {
              $keysString .= ($keysString ? "\n" : '') . $item['product']['name'] . ' - ' . $item['key'];
            }
            
            $this->db->query("UPDATE `" . DB_PREFIX . "order_product` SET `keybin_key` = '".$this->db->escape($item['key'])."' WHERE `order_id` = '".(int) $order_id."' AND `model` = '".(int) $item['product']['id']."'");
          } else {
            $allKeysReady = false;
          }
        }
      }
      
      if ($allKeysReady) {
        if ($this->config->get('keybin_logs')) {
          file_put_contents(DIR_LOGS.'keybin.log', 'All keys ready, update order status' . PHP_EOL, FILE_APPEND | LOCK_EX);
        }
        
        return [
          'status' => 'complete',
          'text' => $keysString,
        ];
      } else {
        if ($this->config->get('keybin_logs')) {
          file_put_contents(DIR_LOGS.'keybin.log', 'Not all keys are ready, do not change order status (will be checked further by cron)' . PHP_EOL, FILE_APPEND | LOCK_EX);
        }
        
        return [
          'status' => 'pending',
          'text' => 'Order complete via Keybin purchase API, Keybin order ID reference: '.$response['id'],
        ];
      }
    } else {
      if ($this->config->get('keybin_logs')) {
        if (!empty($response['message'])) {
          file_put_contents(DIR_LOGS.'keybin.log', 'Error: '. $response['message'] . PHP_EOL, FILE_APPEND | LOCK_EX);
          file_put_contents(DIR_LOGS.'keybin.log', 'Request: ' . PHP_EOL . print_r($params, true) . PHP_EOL, FILE_APPEND | LOCK_EX);
        } else {
          file_put_contents(DIR_LOGS.'keybin.log', 'No order ID has been returned for this order' . PHP_EOL, FILE_APPEND | LOCK_EX);
        }
      }
    }
    
    return [
      'status' => 'pending',
      'text' => '',
    ];
  }
  
  public function getBalance($forceRefresh = false) {
    // Use file-based cache instead of session for better persistence
    $cacheFile = DIR_CACHE . 'keybin_balance.cache';
    $cacheTimeout = 300; // 5 minutes
    
    // Try to read from file cache first
    if (!$forceRefresh && file_exists($cacheFile)) {
      $cacheData = json_decode(file_get_contents($cacheFile), true);
      
      if ($cacheData && isset($cacheData['timestamp']) && isset($cacheData['data'])) {
        $cacheAge = time() - $cacheData['timestamp'];
        
        if ($cacheAge < $cacheTimeout) {
          if ($this->config->get('keybin_logs')) {
            $timeLeft = $cacheTimeout - $cacheAge;
            file_put_contents(DIR_LOGS.'keybin.log', 'Using cached balance from file (expires in '.$timeLeft.' seconds) - Balance: ' . $cacheData['data']['balance'] . PHP_EOL, FILE_APPEND | LOCK_EX);
          }
          
          // Also update session for backwards compatibility
          if (isset($cacheData['data']['balance'])) {
            $balance = $cacheData['data']['balance'] . ' ' . str_replace('EUR', '€', $cacheData['data']['currency']);
            $this->session->data['keybin_balance'] = $balance;
            $this->session->data['keybin_balance_raw'] = $cacheData['data'];
            $this->session->data['keybin_lastupdate'] = $cacheData['timestamp'];
          }
          
          return $cacheData['data'];
        }
      }
    }
    
    // Cache expired or force refresh, call API
    if ($this->config->get('keybin_logs')) {
      $cacheStatus = 'Cache status: ';
      if (!file_exists($cacheFile)) {
        $cacheStatus .= 'No cache file';
      } else {
        $cacheData = json_decode(file_get_contents($cacheFile), true);
        if ($cacheData && isset($cacheData['timestamp'])) {
          $age = time() - $cacheData['timestamp'];
          $cacheStatus .= 'Age=' . $age . 's (expired)';
        } else {
          $cacheStatus .= 'Invalid cache data';
        }
      }
      file_put_contents(DIR_LOGS.'keybin.log', 'Fetching fresh balance from API - ' . $cacheStatus . PHP_EOL, FILE_APPEND | LOCK_EX);
    }
    
    $response = $this->api('account/balance');
    
    // Handle throttle error - return cached data if available
    if (isset($response['statusCode']) && $response['statusCode'] == 429) {
      if ($this->config->get('keybin_logs')) {
        file_put_contents(DIR_LOGS.'keybin.log', 'API throttled (429), attempting to use stale cache' . PHP_EOL, FILE_APPEND | LOCK_EX);
      }
      
      // Return cached data even if expired, better than nothing
      if (file_exists($cacheFile)) {
        $cacheData = json_decode(file_get_contents($cacheFile), true);
        if ($cacheData && isset($cacheData['data'])) {
          if ($this->config->get('keybin_logs')) {
            file_put_contents(DIR_LOGS.'keybin.log', 'Using stale cache due to throttle - Age: ' . (time() - $cacheData['timestamp']) . 's' . PHP_EOL, FILE_APPEND | LOCK_EX);
          }
          return $cacheData['data'];
        }
      }
      
      // Also try session cache
      if (!empty($this->session->data['keybin_balance_raw'])) {
        return $this->session->data['keybin_balance_raw'];
      }
    }
    
    if (isset($response['balance'])) {
      $balance = $response['balance'] . ' ' . str_replace('EUR', '€', $response['currency']);
      
      // Save to file cache
      $cacheData = array(
        'timestamp' => time(),
        'data' => $response
      );
      file_put_contents($cacheFile, json_encode($cacheData));
      
      // Also save to session for backwards compatibility
      $this->session->data['keybin_balance'] = $balance;
      $this->session->data['keybin_balance_raw'] = $response;
      $this->session->data['keybin_lastupdate'] = time();
      
      if ($this->config->get('keybin_logs')) {
        file_put_contents(DIR_LOGS.'keybin.log', 'Balance cached successfully to file - Balance: ' . $response['balance'] . PHP_EOL, FILE_APPEND | LOCK_EX);
      }
    }
    
    return $response;
  }
  
  public function api($method, $params = array()) {
    if (!$this->config->get('keybin_token')) {
      die('keybin token not defined!');
    }
    
    $payload = '';
    
    if (!empty($params)) {
      $payload = json_encode($params);
    }
    
    $curl = curl_init();
    
    curl_setopt_array($curl, array(
      CURLOPT_URL => 'https://api.keybin.net/v1/'.$method,
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_MAXREDIRS => 10,
      CURLOPT_TIMEOUT => 0,
      CURLOPT_FOLLOWLOCATION => true,
      CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
      CURLOPT_HTTPHEADER => array(
        'Content-Type: application/json',
        'PERSONAL-TOKEN:'.$this->config->get('keybin_token'),
      ),
    ));
    
    if (!empty($payload)) {
      curl_setopt($curl, CURLOPT_POST, true);
      curl_setopt($curl, CURLOPT_POSTFIELDS, $payload);
    }

    $response = curl_exec($curl);
    
    curl_close($curl);
    
    $response = json_decode($response, true);
    
    return $response;
  }
  
  private function isKeybinProduct($product_id, $returnListing = false) {
    $product = $this->db->query("SELECT DISTINCT * FROM " . DB_PREFIX . "product WHERE product_id = '" . (int)$product_id . "'")->row;
    
    if (isset($product['import_batch']) && strpos(strtolower($product['import_batch']), 'keybin') !== false) {
      if ($returnListing) {
        if (!empty($product['keybin_listing'])) {
          return $product['keybin_listing'];
        } else {
          if ($this->config->get('keybin_logs')) {
            file_put_contents(DIR_LOGS.'keybin.log', 'Error: Product defined as keybin one but no listingId have been found in product table for product_id = '. $product_id . PHP_EOL, FILE_APPEND | LOCK_EX);
            
            return false;
          }
        }
      }
      
      return true;
    }
    
    return false;
  }
}